<?php
/* --------------------------------------------------------------
 SmartyPluginLoader.php 2020-04-16
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Admin\Layout\Smarty\Loaders;

use Closure;
use Gambio\Core\Application\ValueObjects\UserPreferences;
use Gambio\Core\Language\TextManager;
use Smarty;
use Smarty_Internal_Template;
use Smarty_Resource_Custom;
use SmartyException;

/**
 * Class SmartyPluginLoader
 * @package Gambio\Admin\Layout\Smarty\Loaders
 */
class SmartyPluginLoader implements LoaderInterface
{
    /**
     * @var TextManager
     */
    private $textManager;
    
    /**
     * @var UserPreferences
     */
    private $userPreferences;
    
    
    /**
     * SmartyPluginLoader constructor.
     *
     * @param TextManager     $textManager
     * @param UserPreferences $userPreferences
     */
    public function __construct(TextManager $textManager, UserPreferences $userPreferences)
    {
        $this->textManager     = $textManager;
        $this->userPreferences = $userPreferences;
    }
    
    
    /**
     * @inheritDoc
     */
    public function load(Smarty $smarty): void
    {
        $smarty->registerResource('get_usermod', $this->getUserMod());
        try {
            $smarty->registerPlugin('function', 'load_language_text', $this->loadLanguageText());
        } catch (SmartyException $e) {
        }
    }
    
    
    /**
     * Smarty get_usermod resource callback.
     *
     * @return Smarty_Resource_Custom
     */
    private function getUserMod(): Smarty_Resource_Custom
    {
        return new class extends Smarty_Resource_Custom {
            protected function fetch($name, &$source, &$mtime)
            {
                $path     = realpath(__DIR__ . '/../../../../admin/html/content');
                $filename = "{$path}/{$name}";
                
                $source = file_get_contents($filename);
                $mtime  = filemtime($filename);
            }
        };
    }
    
    
    /**
     * Load language text smarty callback function.
     *
     * @return Closure
     */
    private function loadLanguageText()
    {
        return function (array $params, Smarty_Internal_Template $smarty) {
            $section = $params['section'] ?? '';
            $name    = $params['name'] ?? 'txt';
            
            $sectionArray = $this->textManager->getSectionPhrases($section, $this->userPreferences->languageId());
            $smarty->assign($name, $sectionArray);
        };
    }
}